package it.kqi.chaka095.demo;
import java.rmi.RemoteException;
import java.util.Calendar;

import org.apache.axis2.AxisFault;
import org.apache.axis2.databinding.types.NMTokens;
import org.apache.axis2.databinding.types.PositiveInteger;
import org.apache.axis2.databinding.types.Token;
import org.apache.axis2.databinding.types.URI;
import org.apache.axis2.databinding.types.URI.MalformedURIException;

import it.kqumg.soap.chaka095.http.*;

/**
 *  SMSSender consente di inviare messaggi SMS di tipo semplice o di tipo "personalizzato".
 *  L'implementazione dei vari metodi è basata sulle classi generate dal tool wsdl2java da Axis2.
 *  Lo scopo di questa classe è quella nascondere alcune complessità dell'interfaccia SOAP. 
  */
public class SMSSender {
	private class StartOptions{
		
		public final String userid;
		public final String password;
		public final String contactPerson;
		public final String callbackURL;
		
		public StartOptions(String userid, String password,
				String contactPerson, String callbackURL) {
			super();
			this.userid = userid;
			this.password = password;
			this.contactPerson = contactPerson;
			this.callbackURL = callbackURL;
		}
		
		
	}
	
	private StartOptions startOptions;
	private Token		   authToken;
	private ChakaStub 	   client;
	private String		   senderString;
	
	
	/**
	 * @param userid
	 * @param password
	 * @param contactPerson
	 * @param callbackURL
	 * @throws AxisFault
	 * 
	 * Inizializza un'istanza di SMSSender con i dettagli per autenticazione.
	 * Il parametro callbackURL può essere null se non si desidera sfruttare la funzionlità di callback.
	 * Diversamente deve puntare all'endpoint (http, non https) del servizio Chaka_CB (v. manuale).
	 */
	public SMSSender(String userid, String password, String contactPerson, String callbackURL ) throws AxisFault {
		startOptions = new StartOptions(userid, password, contactPerson, callbackURL);
		client = new ChakaStub();
	}
	
	/**
	 * @param authToken
	 * @throws AxisFault
	 * 
	 * Inizializza un'istanza di SMSSender con il token di autenticazione. Questo consente di 
	 * utilizzare una sessione precedentemente aperta (risparmiando, così una chiamata al metodo SOAP "start"). 
	 * Il metodo getAuthToken consente di ricavare il token corrente (al fine di salvarlo per successivi utilizzi). 
	 */
	public SMSSender( Token authToken ) throws AxisFault {
		this.authToken = authToken;
		this.client = new ChakaStub();
	}
	
	/**
	 * @throws MalformedURIException
	 * @throws RemoteException
	 * 
	 * Inizia una sessione invocando il metodo SOAP "start". Il token viene salvato in authToken.
	 * Questo metodo viene implicitamente chiamato da sendMessage, se authToken non è
	 * ancora stato valorizzato.
	 */
	public void startSession() throws MalformedURIException, RemoteException {
		
		ChakaStub.Start startArgs = new ChakaStub.Start();
		startArgs.setUserid(this.startOptions.userid);
		startArgs.setPassword(this.startOptions.password);
		ChakaStub.UserOptions userOptions = new ChakaStub.UserOptions();
		userOptions.setContactPerson(new URI(this.startOptions.contactPerson));
		if ( this.startOptions.callbackURL != null) {
				userOptions.setSoapCallback(new URI(this.startOptions.callbackURL));
		}
		startArgs.setOptions(userOptions);
		ChakaStub.StartResponse startResp = this.client.start(startArgs);
		this.authToken = startResp.getAuth();
 	}
	
	
Token getAuthToken() { return this.authToken; }
	
	/**
	 *  La classe Destination rappresenta una semplice destinazone SMS.
	 */
	public static class Destination {
		public Destination(String address, String cc1, String cc2) {
			super();
			this.address = address;
			this.cc1 = cc1;
			this.cc2 = cc2;
		}
		public String address;
		public String cc1;
		public String cc2;
	}
	
	/**
	 * La classe DestinationWithMessage rappresenta una destinazione associata a un testo.
	 * Viene usata per creare spedizioni in cui ogni destinazione è accompagnata da un
	 * messaggio diverso (vedi secondo override di sendMessage). 
	 */
	public static class DestinationWithMessage extends Destination {
		public String shortMessage;

		public DestinationWithMessage(String address, String cc1, String cc2,
				String shortMessage) {
			super(address, cc1, cc2);
			this.shortMessage = shortMessage;
		}
	}

	
	/**
	 * @param reqUID
	 * @param subject
	 * @param text
	 * @param destinations
	 * @return
	 * @throws MalformedURIException
	 * @throws RemoteException
	 * 
	 * Questa versione di sendMessage crea una spedizione SMS in cui ogni 
	 * destinatario riceve lo stesso messaggio.
	 * Questo metodo effettua la chiamata SOAP "sendMessage".
	 * Il valore restituito è il codice del messaggio ("msg-id")
	 * 
	 */
	PositiveInteger sendMessage( String reqUID, String subject, String text, Destination[] destinations, java.util.Calendar scheduling ) throws MalformedURIException, RemoteException {
		if ( getAuthToken() == null ) { startSession();}
		
		ChakaStub.DestinationList destList = new ChakaStub.DestinationList();
		/*
		 * senderString è usato per specificare il "mittente testuale" del messaggio.
		 * Se è impostato, poiché il gateway SOAP non supporta esplicitamente
		 * la selezione del mittente testuale, viene impostato il flag SMS_SENDER_IS_A1
		 * e il mittente viene inserito nell'attributo "a1" di ogni destinazione 
		 */
		NMTokens flags = this.senderString != null ?
				new NMTokens("SMS_SENDER_IS_A1") 	:
				new NMTokens("") ;
		for (Destination destination : destinations) {
			ChakaStub.Destination dest = new ChakaStub.Destination();
			dest.setAddress(destination.address);
			dest.setCc1(destination.cc1);
			if ( this.senderString != null ) {
				ChakaStub.DestinationExtraAttributes extraAttrs = new ChakaStub.DestinationExtraAttributes();
				extraAttrs.setA1(this.senderString);
				dest.setExtra(extraAttrs);
			} else {
				dest.setCc2(destination.cc2);
			}
			destList.addItem(dest);
		}
		
		ChakaStub.SMSMessage smsMsg = new ChakaStub.SMSMessage();
		smsMsg.setShortMsg( text );
		
		/*
		 * Il tipo SMSMessageChoice_type0 è generato da Axis2 per controllare i vari modi
		 * in cui l'elemento SOAP di tipo "choice" interno a SMSMessagge può essere valorizzato.
		 * Nel nostro caso, viene impostato "destinations".
		 * 
		 *  Le altre classi che terminano in "Choice_type0" svolgono un ruolo analogo nelle
		 *  altre strutture SOAP che contengono elementi "choice"
		 */
		ChakaStub.SMSMessageChoice_type0 smsMsgChoice0 = new ChakaStub.SMSMessageChoice_type0();
		smsMsgChoice0.setDestinations( destList );
		
		smsMsg.setSMSMessageChoice_type0( smsMsgChoice0 );
		
		ChakaStub.MessageChoice_type0 messageChoice0 = new ChakaStub.MessageChoice_type0();
		messageChoice0.setSms(smsMsg);
		
		ChakaStub.Message message = new ChakaStub.Message();
		message.setMessageChoice_type0(messageChoice0);
		message.setSubject(subject);
		ChakaStub.MessageOptions opts = new ChakaStub.MessageOptions();
		opts.setFlags(flags);
        if ( scheduling != null ) {
                    opts.setScheduling(scheduling);
        }
		message.setOptions(opts);
		
		return this.sendChakaMessage(reqUID, message);
	}
	
	/**
	 * @param reqUID
	 * @param subject
	 * @param destinations
	 * @return
	 * @throws MalformedURIException
	 * @throws RemoteException
	 * 
	 * Questa versione di sendMessage crea una spedizione SMS in cui ogni 
	 * destinatario riceve un messaggio diverso. Il testo dei vari messaggi
	 * sono specificati nell'attributo shortmsg di ogni elemento di desttinations.
	 * Questo metodo effettua la chiamata SOAP "sendMessage".
	 * Il valore restituito è il codice del messaggio ("msg-id")
	 */
	PositiveInteger sendMessage( String reqUID, String subject, DestinationWithMessage[] destinations, java.util.Calendar scheduling ) throws MalformedURIException, RemoteException {
		if ( getAuthToken() == null ) { startSession(); }
		
		/*
		 * Poiché il gateway SOAP non supporta esplicitamente la possibilità di specificare messaggi
		 * distinti per le varie destinazioni, vengono impostati i flag SMS_MESSAGE_IS_A0 UMG_SMSMERGE,
		 * e il testo del messaggio viene specificato nell'attributo "a0" di ogni destinazione.
		 */
		ChakaStub.DestinationList destList = new ChakaStub.DestinationList();
		final String baseflags = "SMS_MESSAGE_IS_A0 UMG_SMSMERGE"; 
		NMTokens flags = this.senderString != null ?
				new NMTokens(baseflags + " SMS_SENDER_IS_A1") 	:
				new NMTokens(baseflags) ; 
		for (DestinationWithMessage destination : destinations) {
			ChakaStub.Destination dest = new ChakaStub.Destination();
			dest.setAddress(destination.address);
			dest.setCc1(destination.cc1);
			dest.setCc2(destination.cc2);

			ChakaStub.DestinationExtraAttributes extraAttrs = new ChakaStub.DestinationExtraAttributes();
			extraAttrs.setA0(destination.shortMessage);
			if ( this.senderString != null ) {
				extraAttrs.setA1(this.senderString);
			}
			dest.setExtra(extraAttrs);
			destList.addItem(dest);
		}
		
		ChakaStub.SMSMessage smsMsg = new ChakaStub.SMSMessage();
		smsMsg.setShortMsg( " " );
		
		/*
		 * (Per il significato dei vari tipi *Choice_type0 vedere commento sopra)
		 */
		ChakaStub.SMSMessageChoice_type0 smsMsgChoice0 = new ChakaStub.SMSMessageChoice_type0();
		smsMsgChoice0.setDestinations( destList );
		
		smsMsg.setSMSMessageChoice_type0( smsMsgChoice0 );
		
		ChakaStub.MessageChoice_type0 messageChoice0 = new ChakaStub.MessageChoice_type0();
		messageChoice0.setSms(smsMsg);
		
		ChakaStub.Message message = new ChakaStub.Message();
		message.setMessageChoice_type0(messageChoice0);
		message.setSubject(subject);
		ChakaStub.MessageOptions opts = new ChakaStub.MessageOptions();
		opts.setFlags(flags);
        if ( scheduling != null ) {
                    opts.setScheduling(scheduling);
        }
		message.setOptions(opts);
		return this.sendChakaMessage(reqUID, message);
	}


	private PositiveInteger sendChakaMessage (String reqUID, ChakaStub.Message message ) throws RemoteException {
		ChakaStub.SendMessage sendMessageArgs = new ChakaStub.SendMessage();
		sendMessageArgs.setAuth(this.authToken);
		sendMessageArgs.setRequestUid(new Token(reqUID));
		sendMessageArgs.setMessage(message);
	    ChakaStub.SendMessageResponse sendResp = this.client.sendMessage(sendMessageArgs);
		return sendResp.getResult().getInfo().getMsgId() ;
	}


	public void setSenderString(String str) {
		this.senderString = str;
	}
	
}
