﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Chaka.ChakaDevelReference;
using System.Xml.Schema;
using System.ServiceModel;

namespace Chaka
{
    public class SMSSender
    {
        public SMSSender(String serviceURL, String userid, String password, String contactEmail, String callBackURL)
        {
            this._userId = userid;
            this._password = password;
            this._contactEmail = contactEmail;
            this._callbackURL = callBackURL;
            this._serviceURL = serviceURL;
        }

        private String _userId { get; set; }
        private String _password { get; set; }
        private String _contactEmail { get; set; }
        private String _callbackURL { get; set; }
        private String _serviceURL { get; set; }


        private ChakaPortTypeClient Client
        {
            get
            {
                if(_client == null)
                    _client = new ChakaPortTypeClient(new BasicHttpBinding(), new EndpointAddress(_serviceURL));

                return _client;
            }
        }

        private ChakaPortTypeClient _client;
        private String _authToken;
        private String _smsSender;

        public void SetSmsSender(String sender)
        {
            this._smsSender = sender;
        }

        public void StartSession()
        {
            // Ottengo un token di autorizzazione tramite la funzione "start"
            this._authToken = Client.start(this._userId, this._password,
               new ChakaDevelReference.UserOptions()
               {
                   contactperson = this._contactEmail,
                   soapcallback = this._callbackURL
               });
        }

        public String getAuthToken() { return _authToken; }

        /// <param name="reqUID">Id della richiesta (deve essere univoco)</param>
        /// <returns>Restituisce il numero di correlazione del messaggio inviato al sistema</returns>
        public String SendMessage(String reqUID, String subject, String text, List<Destination> destinations)
        {
            // Se la sessione non è stata inizializzata all'esterno 
            // provvedo a farlo qui
            if (getAuthToken() == null) { StartSession(); }

            // Imposto il flag "SMS_SENDER_IS_A1" se è stato specificato 
            // un mittente personalizzato
            var flags = (_smsSender != null) ? "SMS_SENDER_IS_A1" : "";


            // Istanzio la lista delle destinazioni (anche se il destinatario è uno solo)
            var destinationList = new DestinationList();
            var dl = new List<Destination>();

            // Preparo la lista delle destinazioni.
            // Se è stato specificato un mittente personalizzato
            // provvedo a impostarlo nel campo cc2.
            // "address" è il numero di telefono a cui verrà spedito il messaggio
            foreach (var destination in destinations)
            {
                var dest = new Destination();
                dest.address = destination.address;
                dest.cc1 = destination.cc1;
                dest.cc2 = destination.cc2;

                var daa = new DestinationExtraAttributes() { a1 = _smsSender };
                dest.extra = daa;
                dl.Add(dest);
            }
            destinationList.item = dl.ToArray();

            // Creo l'sms da spedire e gli associo la lista di destinatari
            // e il testo del messaggio
            SMSMessage smsMessage = new SMSMessage();
            smsMessage.shortmsg = text;
            smsMessage.Item = destinationList;

            // Creo un messaggio contenitore da spedire al sistema e associo l'sms
            var message = new ChakaDevelReference.Message();

            // Specifico il subject, che sarà visibile nel sito di Status
            message.subject = subject;
            message.Item = smsMessage;
            message.ItemElementName = ItemChoiceType.sms;
            
            // Associo i flag di configurazione definiti all'inizio della funzione
            var options = new MessageOptions();
            options.flags = flags;
            message.options = options;

            // Procedo all'invio del messaggio al sistema restituendo il numero di correlazione
            return this.SendChakaMessage(reqUID, message);
        }

        /// <param name="reqUID">Id della richiesta (deve essere univoco)</param>
        /// <returns>Restituisce il numero di correlazione del messaggio inviato al sistema</returns>
        public String SendMessage(String reqUID, String subject, List<DestinationWithMessage> destinations)
        {
            // Se la sessione non è stata inzializzata all'esterno 
            // provvedo a farlo qui
            if (getAuthToken() == null) { StartSession(); }


            // Imposto il flag SMS_SENDER_IS_A1 se non è stato specificato 
            // un mittente personalizzato e specifico con "SMS_MESSAGE_IS_A0 UMG_SMSMERGE"
            // che si tratta di un sms merge
            var flags = (_smsSender != null) ? "SMS_SENDER_IS_A1" : "";
            flags = string.Format("SMS_MESSAGE_IS_A0 UMG_SMSMERGE {0}", flags);


            var destinationList = new DestinationList();
            var dl = new List<Destination>();


            // Preparo la lista delle destinazioni.
            // Se è stato specificato un mittente personalizzato provvedo a impostarlo nel campo cc2.
            // "address" è il numero di telefono a cui verrà spedito il messaggio.
            foreach (var destination in destinations)
            {
                var dest = new Destination();
                dest.address = destination.address;
                dest.cc1 = destination.cc1;
                dest.cc2 = destination.cc2;

                // In questo caso, essendo un sms merge, ogni destinatario contiene un testo personalizzato.
                var daa = new DestinationExtraAttributes() { 
                            a0 = destination.shortMessage,
                            a1 = _smsSender  };
                dest.extra = daa;
                dl.Add(dest);
            }

            destinationList.item = dl.ToArray();

            // Creo l'sms da spedire e gli associo la lista di destinatari
            // e il testo del messaggio
            SMSMessage smsMessage = new SMSMessage();

            // Il testo del messaggio viene specificato su ciascun destinatario,
            // quindi non serve impostarne uno globale
            smsMessage.shortmsg = " ";
            smsMessage.Item = destinationList;

            // Creo un messaggio contenitore da spedire al sistema e associo l'sms
            var message = new ChakaDevelReference.Message();

            // Specifico il subject, che sarà visibile nel sito di Status
            message.subject = subject;
            message.Item = smsMessage;
            message.ItemElementName = ItemChoiceType.sms;

            // Associo i flag di configurazione definiti all'inizio della funzione
            var options = new MessageOptions();
            options.flags = flags;
            message.options = options;

            // Procedo all'invio del messaggio al sistema restituendo il numero di correlazione
            return this.SendChakaMessage(reqUID, message);
        }

        public String SendChakaMessage(String reqUID, Message message)
        {
            // Utilizzo il client creato automaticamente con la reference al servizio soap
            // per effettuare l'invio vero e proprio del messaggio al sistema.
            return this.Client.sendMessage(getAuthToken(), reqUID, message).info.msgid;
        }
    }


    /// <summary>
    /// Classe creata per poter introdurre la proprietà "shortMessage",
    /// contenente il testo del messaggio personalizzato per ogni destinatario
    /// </summary>
    public class DestinationWithMessage : Destination
    {
        public String shortMessage { get; set; }
    }
}
